;;;
;;; Virtual Disk Drive Project - Oct. 22, 2004
;;;
;;;    Apple 2 support
;;;

#include "main.h"
#include "vd.h"
#include "utility.h"
#include "headmove.mac"
#include "waitP1.mac"
#include "disk.h"
#include "apple.mac"
#include "apple.h"

PROG1	CODE
		extern	SerTrackSecDone
		extern	SerTrackFinal

		global	GCRTrailerP1
		global	GCR6x2SectorP1
		global	GCR5x3SectorP1
		global	NIBSectorP1


;;;*******************************************************************
;;; NAME:	GCRProEpiLog	()
;;;
;;; DESCR:	Generates a three-byte prolog or epilog based upon
;;;		either data read from memory, or static values.
;;;
;;;		If "TIGHT" is given, then code will be generated
;;;		that uses more space but gives a couple of extra
;;;		cycles on the tail end.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- used at (-11) and returns at (-13) if "TIGHT", or
;;;			at (-11) otherwise
;;;		- BIG NOTE - the values for one, two, and three cannot
;;;		  crash into the defines for "read", "TIGHT", and "LOOSE"
;;;*******************************************************************
#define READ	0x0100
#define TIGHT	0x0200
#define LOOSE	0x0300
	
GCRProEpiLog:	macro	one,two,three,tight

	;; first pro/epilog byte
	
		if (one == READ)
			MemReadByte				; (-11) first pro/epilog byte
			MemLowInc				; (-10)
		else
			movlw		one			; (-11)
			Wait2					; (-10)
		endif

		Wait2						; (-8)
		call		GCRSimpleByte			; (-6) saves some space with a "call"

	;; second pro/epilog byte
	
		if (two == READ)
			MemReadByte				; (-11) second pro/epilog byte
			MemLowInc				; (-10)
		else
			movlw		two			; (-11)
			Wait2					; (-10)
		endif

		Wait2						; (-8)
		call		GCRSimpleByte			; (-6) saves some space with a "call"
	
	;; third pro/epilog byte (this is where "TIGHT" comes in)
	
		if (three == READ)
			MemReadByte				; (-11) third pro/epilog byte
			MemLowInc				; (-10)
		else
			movlw		three			; (-11)
			Wait2					; (-10)
		endif

		if (tight == TIGHT)
			Wait4					; (-8)
			movwf			GCR_OUT		; (-4)
			GCRSimpleByteMacro			; (-3) saves some cycles this way
			Wait2					; (-3)
			BitPulse				; (-1)
								; (-13)
		else
			Wait2					; (-8)
			call		GCRSimpleByte		; (-6) saves some space with a "call"
								; (-11)
		endif
		endm

		
;;;*******************************************************************
;;; NAME:	GCRShiftBit()
;;;
;;; DESCR:	Called to shift out a single bit from the GCR_OUT buffer
;;;		using GCR format, while shifting pre-mapped data.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- shifts out one bit from GCR_OUT to RDATA
;;;		- shifts a bit in from the pre-GCR buffer GCR_B2
;;;		- loads the pre-GCR buffer GCR_B2 if we run out of bits in it
;;;		- if we hit the end of the incoming bytes, GCR_B2 is cleared
;;;			This causes zero bits to fill "left-over" bits
;;;*******************************************************************
GCRShiftBit:
	BitPulse		; (-1)

	;; assumes that at least one bit is available when this is called

	rlf	GCR_B2		; (-13) rotate into carry bit
	rlf	GCR_B1		; (-12) take carry in

	decfsz	GCR_B2CLEAR	; (-11) used all B2 bits?
	goto	GCRShiftBit1	; (-10) nope, continue

	movlw	0x00		; (-9) prepare for below (clear W)

	decfsz	GCR_BYTECOUNT	; (-8) consumed all bytes?
	MemReadByte		; (-7) nope, read new byte into W

	movwf	GCR_B2		; (-6) move new W (maybe zero) into the B2 buffer

	return			; (-5)

GCRShiftBit1:
	Wait3			; (-8)
	return			; (-5)

;;;*******************************************************************
;;; NAME:	GCRSimpleBit()
;;;
;;; DESCR:	Called to shift out a single bit from the GCR_OUT buffer
;;;		using GCR format.  NO shifting of pre-mapped data.
;;; 
;;;		This routine will also align memory by INCVAR bytes if
;;;		INCVAR is not zero.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	IMPORTANT - this routine doesn't mess with W,
;;;			because BitPulse doesn't, and neither does
;;;			CondMemIncLow
;;;*******************************************************************
GCRSimpleBit:
	BitPulse		; (-1)
	CondMemIncLow		; (-13)
	Wait2			; (-7)
	return			; (-5)

GCRSimpleBitNW:
	BitPulse		; (-1)
	return			; (-13)
	
	
;;;**********************************************************************
;;; NAME:	GCRSimpleByte()
;;;
;;; DESCR:	Send out a simple byte (8 bits).
;;;
;;;		This routine will also align memory by INCVAR bytes if
;;;		INCVAR is not zero.  This capability is provided by
;;;		GCRSimpleBit.  GCRSimpleBit() will increment memory by
;;;		one.  So this routine will do up to 7 increments.
;;; 
;;; ARGS:	W has the byte to send out
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- called at (-6) returns at (-11)
;;;		- IMPORTANT NOTE - this routine doesn't mess with W
;;;				because GCRSimpleBit/NW don't.
;;;		- IMPORTANT NOTE - GCR_OUT gets rotated back to its original
;;;			value during this routine - therefore, it looks
;;;			like it wasn't touched.
;;;**********************************************************************
GCRSimpleByte:
	movwf	GCR_OUT		; (-4)
	GCRSimpleByteMacro	; (-3)
	goto	GCRSimpleBitNW	; (-3) bit 8 (return from this routine at -11)

;;;*******************************************************************
;;; NAME:	GCR4x4Byte()
;;;
;;; DESCR:	Generate a 2-byte output from an incoming byte (in ARG0)
;;;		in 4x4 GCR encoding for the apple.
;;;
;;; ARGS:	W has the byte to encode/output, ARG0 is used
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- 4x4 is simply the odd and even bits:
;;;			Byte   = H G F E D C B A
;;;		===>	Output = 1 H 1 F 1 D 1 B   1 G 1 E 1 C 1 A
;;;		- called at (-10)
;;;		- NOTE - return is from GCRSimpleByte at -11
;;;		- Unfortunately, GCRSimpleByte() is somewhat replicated
;;;		  here because I needed the cycles
;;;*******************************************************************
GCR4x4Byte:

	MemReadByte		; (-8)
	movwf	ARG0		; (-7)
	rrf	ARG0,W		; (-6) prepare odd bits
	iorlw	0xaa		; (-5)
	movwf	GCR_OUT		; (-4)
	GCRSimpleByteMacro	; (-3) need to do this inline
	call	GCRSimpleBitNW	; (-3) last bit here

	movf	ARG0,W		; (-11) prepare for the even bits
	iorlw	0xaa		; (-10)
	movwf	GCR_OUT		; (-9)
	
	MemLowInc		; (-8)
	goto	GCRSimpleByte	; (-6) (return from this routine at -11)

;;;**********************************************************************
;;; NAME:	GCRSelfSync()
;;;
;;; DESCR:	Does the ARG0 count of self-sync "bytes."  These self-
;;;		syncs consists of 8 ones followed by 2 zeros.
;;;
;;;		This routine will also align memory by INCVAR bytes if
;;;		INCVAR is not zero.  This capability is provided by
;;;		GCRSimpleByte, although this is the only routine within
;;;		which INCVAR will be set to non-zero.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- GCRSimpleByte is the routine that notices the INCVAR and
;;;		  performs the memory increment if needed.
;;;		- called at (-9), returns at (-16)
;;;**********************************************************************
GCRSelfSyncLoop:
	Wait10			; (-17)

GCRSelfSyncGap:
	movlw	0xff		; (-7)
	call	GCRSimpleByte	; (-6)

	;; now we just have to wait (no output) for the zero times

	Wait11			; (-11)
	Wait20			; (-0) bit 9 as a zero

	decfsz	ARG0		; (-0) bit 10 as zero, and continue generating sequence
	goto	GCRSelfSyncLoop	; (-19)

	;; looks like the apple expects an 0xff followed by a single zero after sync
	
	movlw	0xff		; (-18)
	Wait11			; (-17)
	call	GCRSimpleByte	; (-6)
	Wait11			; (-11)
	Wait2			; (0) this gives us a zero
	
	return			; (-18)
		
;;;**********************************************************************
;;; NAME:	GCR6x2Sector()
;;;
;;; DESCR:	Generate a complete sector in GCR 6x2 format.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:
;;;		- does a "goto SerTrackSecDone" as its return.  Unlike the
;;;		  other sector routines which are in page 2, this routine
;;;		  runs in page one...so there is no PAGESEL necessary
;;; 
;;; 		- things going out in this routine:
;;;		HEADER
;;;		  (- note that ALL header bytes except for the EB in the
;;;		    trailers are in the sector header.
;;;		    this allows for most weird copy protection schemes.  This
;;;		    means that there are 9 bytes for the header, and 6
;;;		    bytes for the data section - meaning 15 bytes times
;;;		    16 sectors (* 15 16) = 240 or 13 sectors (* 15 13) 195
;;;		    The "filler" for the rest of the disk is just 11's 0xff
;;;		    and when generating the filler, the SVD header will be
;;;		    realigned as necessary (just like in the WD formats).
;;;		 
;;;		  - prolog - D5 AA 96 (standard data - no GCR'ing) (3 bytes in 3 bytes)
;;;		  - volume - XX encoded in 4x4 (even odd) (1 byte in 2 bytes)
;;;		  - track  - XX encoded in 4x4 (even odd) (1 byte in 2 bytes)
;;;		  - sector - XX encoded in 4x4 (even odd) (1 byte in 2 bytes)
;;;		  - checksum - XX encoded in 4x4 (even odd) (1 byte in 2 bytes)
;;;		  - epilog - DE AA EB (standard data - no GCR'ing) (2 bytes in 2 bytes + static EB)
;;;
;;;		DATA
;;;		  - prolog - D5 AA AD (standard data - no GCR'ing) (3 "bytes")
;;;		  - data - 342 bytes (made from 256 bytes GCR 6x2'd) (342 "bytes")
;;;		  - checksum - no encoding (1 bytes)
;;;		  - epilog - DE AA EB (standard data - no GCR'ing) (2 bytes in 2 bytes + static EB)
;;;
;;;		GENERATED DATA
;;;		  - prolog - 3 bytes in 3 bytes
;;;		  - volume - 1 byte in 2 bytes
;;;		  - track  - 1 byte in 2 bytes
;;;		  - sector - 1 byte in 2 bytes
;;;		  - checksum - 1 byte in 2 bytes
;;;		  - epilog - 2 bytes in 2 bytes + static EB
;;;		  - data prolog - 3 bytes
;;;		  - data - 342 bytes 
;;;		  - checksum - 1 bytes
;;;		  - epilog - 2 bytes in 2 bytes + static EB
;;;
;;; Sector Header Format (have 13/16 sectors)
;;;
;;;	- sector type  - 1
;;;	- prolog       - 3
;;;	- volume       - 1
;;;	- track        - 1
;;;	- sector       - 1
;;;	- checksum     - 1
;;;	- epilog       - 2 (static 3rd byte)
;;;	- data prolog  - 3
;;;	- data preload - 1
;;;	- data	       - 256
;;;	- checksum     - 1
;;;	- epilog       - 0 (static epilog)
;;;			------
;;;			15 bytes
;;; 
;;;	With 15 bytes per sector plus one for the trailer type and one for trailer length
;;;	
;;; NOTE - from the 5x3code.txt it appears that the header header for 5x3 is D5 AA B5
;;;		- the header for the data looks right
;;;	- the trailer seems to only be DE AA for 5x3 (for both header & data)
;;; NOTE - the gap lengths are somewhat arbitrary...
;;;**********************************************************************
#define GCR_PRE_ADDR_GAP	d'20'
#define GCR_PRE_DATA_GAP	d'7'
#define GCR_EPILOG1		0xde
#define GCR_EPILOG2		0xaa
#define GCR_EPILOG3		0xeb

GCRSector:	macro	EncodingPrep,EncodingMacro

		movlw		GCR_PRE_ADDR_GAP	; (-11)
		movwf		ARG0			; (-10)
		call		GCRSelfSyncGap		; (-9) generate the gap

	;; note that the index hole doesn't matter for the apple, but it is
	;; brought down for fun...
	
		Lower		INDEX_HOLE		; (-16)
		incf		SECCOUNT		; (-15) prepare for later
		Wait3					; (-14)

	;; get 3 verbatim bytes from the header for prolog

		GCRProEpiLog	READ,READ,READ,LOOSE	; (-11)

	;; get 4 bytes from header that go into GCR4x4 encoding
	;; volume, track, sector, checksum

		Wait1					; (-11)
		call		GCR4x4Byte		; (-10)

		Wait1					; (-11)
		call		GCR4x4Byte		; (-10)

		Wait1					; (-11)
		call		GCR4x4Byte		; (-10)

		Wait1					; (-11)
		call		GCR4x4Byte		; (-10)

	;; now the header epilog - last byte is static
	
		GCRProEpiLog	READ,READ,GCR_EPILOG3,LOOSE  ; (-11)

	;; do the pre-data gap

		Wait2					; (-11)
		movlw		GCR_PRE_DATA_GAP	; (-9)
		movwf		ARG0			; (-8)
		call		GCRSelfSyncGap		; (-7) generate the gap

		Wait2					; (-16)
		EncodingPrep				; (-14) GCR5x3Prep or GCR6x2Prep

	;; get 3 verbatim bytes from the header for data prolog,

		GCRProEpiLog	READ,READ,READ,TIGHT	; (-11)

	;; now we're ready to do the data...the first few bits
	;; are in the header block...so it is the GCRXxXData macro
	;; that flips to the data block

		EncodingMacro				; (-13) GCR5x3Data or GCR6x2Data
	
	;; flip back to the header block

		MemAddrLoSet				; (-13)

	;; send out the data checksum

		Wait2					; (-11)
		MemReadByte				; (-9)
		MemLowInc				; (-8)
		call		GCRSimpleByte		; (-6)
		
	;; do the data epilog - all static

		GCRProEpiLog	GCR_EPILOG1,GCR_EPILOG2,GCR_EPILOG3,LOOSE  ; (-11)

;;; 		PAGESEL		SerTrackSecDone		; (-11) NOT NEEDED HERE
		goto		SerTrackSecDone		; (-11)
		endm

GCR5x3SectorP1:
;;; 	GCRSector	GCR5x3Prep,GCR5x3Data
	
GCR6x2SectorP1:
	GCRSector	GCR6x2Prep,GCR6x2Data
	
	
	
;;;**********************************************************************
;;; NAME:	GCRTrailer()
;;;
;;; DESCR:	Generate a simple GCR trailer...increments memory as needed
;;;		too.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:	- same trailer (except for length) for both 6x2 and 5x3
;;;		- does a "goto SerTrackSecDone" as its return.  Unlike the
;;;		  other sector routines which are in page 2, this routine
;;;		  runs in page one...so there is no PAGESEL necessary
;;;		- the number of bytes chosen for the trailer is rather
;;;		  random...it should be chosen so that rotational tests
;;;		  think that the SVD is rotating at the right speed
;;;**********************************************************************
#define GCR_TRAILER_BYTES	d'50'
	
GCRTrailerP1:
		MemReadByte				; (-11) get the number of trailer inc's needed
		movwf		INCVAR			; (-10) place it in right place to cause inc's
		movlw		GCR_TRAILER_BYTES	; (-9) 
		movwf		ARG0			; (-8)
		call		GCRSelfSyncGap		; (-7)
	
;;;		PAGESEL		SerTrackFinal		; (-16) NOT NEEDED HERE
		goto		SerTrackFinal		; (-16)

;;;*******************************************************************
;;; NAME:	NIBSector
;;;
;;; DESCR:	A "raw" format of nibbles.  Unlike the other sector formats,
;;;		this one "takes over" generation of an entire track.
;;;		It only returns when the disk lines are noticed being
;;;		low.
;;;
;;; ARGS:	
;;;
;;; RETURNS:	
;;;
;;; NOTES:
;;;		- for RAW the assumption is that ANY number of zeros
;;;		  can precede a good byte
;;;*******************************************************************

;;; re-use the GCR registers here for ease
	
#define FF_BYTE_TEST	GCR_B1
#define BYTE_COUNT	GCR_BYTECOUNT
#define DISK_SELECTION	GCR_B2CLEAR

NIBSectorP1:
		clrf		BYTE_COUNT	; will cycle back to zero each time
		incf		SECCOUNT	; first move to sector block #1
		MemAddrInc	SECCOUNT

NIBSectorByte:	;; generate 256 bytes of data out
	
		MemReadByte			; (-9)
		movwf		GCR_OUT		; (-7)
		movwf		FF_BYTE_TEST	; (-6)
		skpnz				; (-5) check for self-sync token (0x00)
		comf		GCR_OUT		; (-4) set it to 0xff
	
NIBBit1:	call		GCRSimpleBitNW	; (-3)

		decf		BYTE_COUNT	; (-11) prepare for next byte
		
	;; check for end of a track, if so, reset sector count
	
		;; this confusing piece of code says (1) check to see if SECCOUNT equals
		;; SECTORS (ie - we're done with a track), (2) if so, then check to see
		;; if the BYTE_COUNT is zero (ie - we're done with the sector), if we are
		;; then (3) reset the sector count.

		movf		SECCOUNT,W	; (-10) if on 26, and done with the sector then
		subwf		SECTORS,W	; (-9)      we're done with the track
		skpnz				; (-8) if NOT done with track, then double skip
		movf		BYTE_COUNT	; (-7) resets the Z flag based upon BYTE_COUNT
		skpnz				; (-6) if NOT end of sector then skip
		clrf		SECCOUNT	; (-5) if end of track, reset sector counter

		Wait1				; (-4)

NIBBit2:	call		GCRSimpleBitNW	; (-3)

	;; increment the sector count if we are at the end of the sector
	;; moves us to either the next sector or sector 1

		movf		BYTE_COUNT	; (-11) 
		skpnz				; (-10)
		incf		SECCOUNT	; (-9)

		MemLowInc			; (-8) increment memory (could be done any place along here)

		Wait3				; (-6)

NIBBit3:	call		GCRSimpleBitNW	; (-3)
	
	;; prepare memory location for the next sector/track if at the end of the sector
	;; if it hasn't changed, then this has no real effect

		MemAddrLoSet			; (-11) reset the base
		MemAddrInc	SECCOUNT	; (-9) and adjust for target sector
		Wait4				; (-7)

NIBBit4:	call		GCRSimpleBitNW	; (-3)

	;; check to see if we are still supposed to be generating data

		;; we will only break out of the loop, however, when we are the end of a
		;; sector...doesn't matter where we are in the track.
		;; NOTE - that there is a potential race condition here, where the
		;;   status of the disk selection may change...not worried about it!

		DiskLineMap			; (-11) prepare for DiskTable call below
		Wait1				; (-4)
	
NIBBit5:	call		GCRSimpleBitNW	; (-3)

		call		DiskTable	; (-11) get the selected disk
		movwf		DISK_SELECTION	; (-4) W now has the selected disk

NIBBit6:	call		GCRSimpleBitNW	; (-3)

		movf		DISK_SELECTION,W; (-11)
		xorwf		CURDISK,W	; (-10) zero indicates we're on the same disk	
		InvertZ				; (-9) now a zero indicates that we have a diff disk (or off)
		skpnz				; (-7)
		movf		BYTE_COUNT	; (-6) only worry about it at end of sector
		skpnz				; (-5)
		goto		NIBSectorReturn	; (-4)
	
NIBBit7:	call		GCRSimpleBitNW	; (-3)
		movf		FF_BYTE_TEST	; (-11) check to see if we have a self-sync
		skpnz				; (-10) if not zero, then have a regular byte
		goto		NIBSector0xff	; (-9) if zero, deal with a self-sync

		Wait5				; (-8)

			
NIBBit8:	call		GCRSimpleBitNW	; (-3)
		goto		NIBSectorByte	; (-11)
		

NIBSector0xff:
		;; generate the last bit of an 0xff along with 2 zeros

		Wait4				; (-7)
		call		GCRSimpleBitNW	; (-3)
		Wait20				; (-11) generate a zero
		Wait20				; (-11) generate another zero
		goto		NIBSectorByte	; (-11)
	
NIBSectorReturn:
		;; disk is no longer selected
		;; (note that we didn't generate the last couple bits...no problem though)
		;; and we are sitting at byte 1 of the 256 byte buffer
		;; we need to increment 254 times, leaving the pointer at the trailer selector
	
		decf		BYTE_COUNT
		decf		BYTE_COUNT
	
NIBSectorAlign:	
		MemLowInc
		decfsz		BYTE_COUNT
		goto		NIBSectorAlign

		MemAddrLoSet			; flip to header

		movf		SECTORS,W
		movwf		SECCOUNT	; point to last sector
	
;;; 		PAGESEL		SerTrackSecDone	; (-7) NOT NEEDED HERE
		goto		SerTrackSecDone	; (-7) 

			
;;; =======================================================================
;;; =======================================================================

	END
